<?php
// Include the database configuration file and header
include 'config.php';

// Create connection
$conn = new mysqli($servername, $username, $password, $dbname);

// Check connection
if ($conn->connect_error) {
    die("Connection failed: " . $conn->connect_error);
}

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    echo "Please log in.";
    exit;
}

// Get the current user's ID and verify their status
$user_id = $_SESSION['user_id'];

// Fetch user status
$status_query = "
    SELECT status, mt5username
    FROM users
    WHERE id = ?
";

$stmt = $conn->prepare($status_query);
$stmt->bind_param("i", $user_id);
$stmt->execute();
$stmt->bind_result($status, $mt5username);
$stmt->fetch();
$stmt->close();

if ($status != 1) {
    echo "Access denied. You do not have the required status.";
    exit;
}

// Fetch the percentage value from the users table
$percentage_query = "
    SELECT percentage
    FROM users
    WHERE mt5username = ?
";

$stmt = $conn->prepare($percentage_query);
$stmt->bind_param("s", $mt5username);
$stmt->execute();
$result = $stmt->get_result();
$percentage_row = $result->fetch_assoc();
$percentage = isset($percentage_row['percentage']) ? (float)$percentage_row['percentage'] : 0;
$stmt->close();

// Fetch user-related data from userinfo table
$queries = [
    "profit_query" => "
        SELECT profit
        FROM userinfo
        WHERE mt5username = ?
    ",
    "balance_query" => "
        SELECT balance 
        FROM userinfo
        WHERE mt5username = ?
    ",
    "trades_open_query" => "
        SELECT open_trades 
        FROM userinfo
        WHERE mt5username = ?
    ",
    "total_invested_query" => "
        SELECT investment 
        FROM users
        WHERE mt5username = ?
    "
];

$results = [];
foreach ($queries as $key => $query) {
    $stmt = $conn->prepare($query);
    $stmt->bind_param("s", $mt5username);
    $stmt->execute();
    $result = $stmt->get_result();
    $results[$key] = $result->fetch_assoc();
    $stmt->close();
}

// Calculate adjusted profit
$profit = isset($results['profit_query']['profit']) ? $results['profit_query']['profit'] : 0;
$adjusted_profit = $profit * (1 - $percentage / 100);
$balance = isset($results['balance_query']['balance']) ? $results['balance_query']['balance'] : 0;
$adjusted_balance = $balance * (1 - $percentage / 100);
$trades_open = isset($results['trades_open_query']['open_trades']) ? $results['trades_open_query']['open_trades'] : 0;
$total_invested = isset($results['total_invested_query']['investment']) ? $results['total_invested_query']['investment'] : 0;
$conn->close();

// Function to format numeric values
function formatValue($value) {
    return is_numeric($value) ? number_format($value, 2) : '0.00';
}

// Function to format integer values
function formatInteger($value) {
    return is_numeric($value) ? number_format((int)$value) : '0';
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard</title>
    <style>
    /* Table styling */
    table {
        width: 90%;
        margin: 0 auto;
        border-collapse: collapse;
        background-color: rgba(0, 0, 0, 0.5);
        border-radius: 8px;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.7);
        overflow: hidden;
        color: #d4af37;
    }

    /* Table header and cell styling */
    td {
        padding: 15px;
        text-align: left;
        color: #d4af37;
        font-size: 20px;
        border-bottom: 1px solid #6f6242;
    }

    tr:nth-child(even) {
        background-color: rgba(21, 37, 37, 0.3);
    }

    tr:nth-child(odd) {
        background-color: rgba(0, 0, 0, 0.3);
    }

    td:first-child {
        font-weight: bold;
        background-color: rgba(21, 37, 37, 0.6);
        color: #b2ac8a;
    }

    td span {
        font-weight: 500;
        color: #d7d1ac;
        font-size: 20px;
    }

    tr:hover {
        background-color: rgba(13, 64, 49, 0.5);
        transition: background-color 0.3s ease-in-out;
    }

    @media (max-width: 768px) {
        table {
            width: 90%;
        }

        td {
            font-size: 14px;
            padding: 10px;
        }

        td span {
            font-size: 20px;
        }
    }
    
    </style>
</head>
<body>
    <br>
    <table>
        <tbody>
            <tr>
                <td>Investment</td>
                <td>£<span><?php echo formatValue($total_invested); ?></span></td>
            </tr>
            <tr>
                <td>Account Balance</td>
                <td>£<span><?php echo formatValue($adjusted_balance); ?></span></td>
            </tr>
            <tr>
                <td>Trades Open</td>
                <td><span><?php echo formatInteger($trades_open); ?></span></td>
            </tr>
            <tr>
                <td>Percentage</td>
                <td><span><?php echo formatValue($percentage); ?>%</span></td>
            </tr>
        </tbody>
    </table>
    <br>
</body>
</html>
